////////////////////////////////////////////////////////////////////////////////
/*!
  @file   deferred_frag.glsl
  @author Tommy A. Brosman IV
  @par    email:  tbrosman\@digipen.edu
  @date   February 14, 2009

  @brief
    A shader for the final render target.
*/
////////////////////////////////////////////////////////////////////////////////

// useful links:
// http://www.gamerendering.com/2009/01/14/ssao/
// http://www.gamedev.net/community/forums/topic.asp?topic_id=463075

#pragma optimize(on)
#pragma optionNV(fastmath on)
#pragma optionNV(fastprecision on)
#pragma optionNV(ifcvt none)
#pragma optionNV(inline all)
#pragma optionNV(strict on)
#pragma optionNV(unroll all)

uniform vec4 cameraLightVec;
uniform sampler2D geomTex;
uniform sampler2D depthTex;
uniform sampler2D normalTex;
uniform sampler2D shadowTex;
uniform sampler2D normalNoiseTex;
uniform sampler2D finalShadowTex;

#define SAMPLES 12 // 10 is good
const float invSamples = 1.0/float(SAMPLES);

// there is a ray extending from the screen-space eye point (0,0,0) to the point
// specified by uv and the depth at uv
varying vec2 uv;

const float radius = 0.00015;
const float weight = 0.07;
const float noiseScale1 = 60.0;
const float noiseScale2 = 40.0;

// these are the random vectors inside a unit sphere
vec3 pSphere[16] = vec3[](
  vec3(0.53812504, 0.18565957, -0.43192),
  vec3(0.13790712, 0.24864247, 0.44301823),
  vec3(0.33715037, 0.56794053, -0.005789503),
  vec3(-0.6999805, -0.04511441, -0.0019965635),
  vec3(0.06896307, -0.15983082, -0.85477847),
  vec3(0.056099437, 0.006954967, -0.1843352),
  vec3(-0.014653638, 0.14027752, 0.0762037),
  vec3(0.010019933, -0.1924225, -0.034443386),
  vec3(-0.35775623, -0.5301969, -0.43581226),
  vec3(-0.3169221, 0.106360726, 0.015860917),
  vec3(0.010350345, -0.58698344, 0.0046293875),
  vec3(-0.08972908, -0.49408212, 0.3287904),
  vec3(0.7119986, -0.0154690035, -0.09183723),
  vec3(-0.053382345, 0.059675813, -0.5411899),
  vec3(0.035267662, -0.063188605, 0.54602677),
  vec3(-0.47761092, 0.2847911, -0.0271716)
);

// (dx, dy, r^2)
vec3 pSquare[12] = vec3[](
  vec3(+0.0, +1.0, +1.0),
  vec3(+0.0, -1.0, +1.0),
  vec3(+1.0, +0.0, +1.0),
  vec3(-1.0, +0.0, +1.0),
  
  vec3(+1.0, +1.0, +2.0),
  vec3(-1.0, -1.0, +2.0),
  vec3(+1.0, -1.0, +2.0),
  vec3(-1.0, +1.0, +2.0),
  vec3(+2.0, +0.0, +2.0),
  vec3(-2.0, +0.0, +2.0),
  vec3(+0.0, -2.0, +2.0),
  vec3(+0.0, +2.0, +2.0)
);

#define NEAR_PLANE 0.1
#define FAR_PLANE 300.0
//#define SCREEN_WIDTH 1200.0
//#define SCREEN_HEIGHT 900.0

mat2 texelToScreen = mat2(SCREEN_WIDTH/2.0, 0.0, SCREEN_HEIGHT/2.0, 0.0);
mat2 screenToTexel = mat2(2.0/SCREEN_WIDTH, 0.0, 2.0/SCREEN_HEIGHT, 0.0);
vec4 sampleToFloat = vec4(1.0, 1.0 / 256.0, 1.0 / 65536.0, 1.0 / 16777216.0);

vec2 TexelToScreen(in vec2 texCoord)
{
  return vec2((SCREEN_WIDTH / 2.0) * texCoord.x, (SCREEN_HEIGHT / 2.0) * texCoord.y);
}

vec2 ScreenToTexel(in vec2 texCoord)
{
  return vec2((2.0 / SCREEN_WIDTH) * texCoord.x, (2.0 / SCREEN_HEIGHT) * texCoord.y);
}

float SampleToFloat(in vec4 sample)
{
  return sample.r + (sample.g / 256.0) + (sample.b / 65536.0) + (sample.a / 16777216.0);
}

vec3 Saturate(in vec3 color, in float factor)
{
  return vec3(pow(color.r, factor + 1.0), pow(color.g, factor + 1.0), pow(color.b, factor + 1.0));
}

void main(void)
{
	float fragDepth     = SampleToFloat(texture2D(depthTex, uv.xy));
	vec3  fragPos       = vec3(uv.xy, fragDepth);
	vec3  fragNorm      = 2.0*texture2D(normalTex, uv.xy).xyz - vec3(1.0);

	// generate a random normal
	vec3  randNorm      = normalize(2.0*texture2D(normalNoiseTex, noiseScale1*uv*fragNorm.xy).xyz - vec3(1.0));
  //vec3 randNorm = normalize(vec3(1.0, 1.0, 0.0));

	float blockingFactor = 0.0;

	vec3 randRay, test;
	float zDiff, normDiff;
	float rayScale = (radius / (fragDepth + 0.01));
	//float rayScale = 30.0*radius;
	float distMult = 1.0;

	for(int i = 0; i < SAMPLES; i+=2)
	{ 
	  distMult = pSquare[i].z;

	  // generate the actual point and flip the ray if it is outside the
	  // hemisphere defined by the fragment normal and the radius
	  //test = fragPos + distMult*rayScale*sign(dot(randRay, fragNorm))*randRay;
	  test.xy = fragPos.xy + (screenToTexel * pSquare[i].xy);
	  
	  // calculate the linear z difference between the end of the ray and the
 	  // fragment to see if the fragment occludes it
	  zDiff = fragDepth - dot(sampleToFloat, texture2D(depthTex, test.xy));
	  
	  // avoid negative z differences, adjust for small differences
	  zDiff = step(0.0, zDiff) * sqrt(clamp(zDiff, 0.0, 1.0));

	  // calculate a weight between the normal at the current fragment and the
	  // normal at the test fragment
	  normDiff = 1.0 - dot(2.0*texture2D(normalTex, test.xy).xyz - vec3(1.0), fragNorm);
	  
	  blockingFactor += (weight / distMult) * smoothstep(0.01, 0.4, normDiff + 20.0*zDiff);
	}

  float nL = smoothstep(0.0, 1.0, dot(normalize(fragNorm), normalize(cameraLightVec.xyz)));

  /*
  vec3 halfVec = normalize(normalize(cameraLightVec.xyz) + vec3(0.0, 0.0, 1.0));
  float specular = smoothstep(0.0, 1.0, pow(dot(halfVec, norm), 5));
  */

	// Change output if at the far plane
	float atFarPlane = step(0.9, fragDepth);

	//gl_FragColor.rgb = vec3(1.0 - blockingFactor);
	
	/*
	// Accumulate shadow samples
	float shadowPercent = 0.0;
	float sum = 0.0;
	float shadowScale = 1.0;
	float angle = 0.5 + 0.5*fragNorm.x;
	mat2 rotMat = mat2(cos(angle), -sin(angle), sin(angle), cos(angle));
	
	for(int i = 0; i < 8; i++)
	{
	  test.xy = uv.xy + (screenToTexel * shadowScale * rotMat * pSquare[i].xy);
	  shadowPercent += pSquare[i].z * SampleToFloat(texture2D(finalShadowTex, test.xy));
	  sum += pSquare[i].z;
	}
	
	shadowPercent /= sum;
	*/
	
	vec3 colorSum = texture2D(geomTex, uv.xy).rgb;
	/*
	for(int i = 0; i < 4; i++)
	{
	  colorSum += texture2D(geomTex, uv.xy + screenToTexel * pSquare[i].xy).rgb;
	}
	
	colorSum = colorSum / 7.0;
	*/
	
  float lightCoeff = min((1.0 - blockingFactor) * (0.35 + 0.8*nL), 1.0);
	vec3 color = Saturate(lightCoeff * colorSum, 0.7*(1.0 - lightCoeff));
	
	gl_FragColor.rgb = mix(color, vec3(0.5, 0.5, 0.5), atFarPlane); // vec3(SampleToFloat(texture2D(shadowTex, uv.xy)));
	//gl_FragColor.rgb = shadowPercent * texture2D(geomTex, uv.xy).rgb;
	gl_FragColor.a = 1.0;
}
